import { env } from "@/env";
import { getCurrentSessionFromServer } from "@/features/auth/lib/session.amplify";
import { validatedFetch } from "@/lib/zod/validated-fetch";
import { otpFeedStatusesResponseSchema } from "@/zod/feeds.schema";
import { NextRequest } from "next/server";

const INTERNAL__NotAuthorizedError = "/api/statuses: Unauthorized - No bearer token found for current session.";

export async function GET(_req: NextRequest): Promise<Response> {
  const agencies = await getAgencies()

  if (agencies instanceof Error) {
    return new Response(JSON.stringify({ error: agencies.message, stack: agencies.stack }), {
      status: INTERNAL__NotAuthorizedError === agencies.message ? 401 : 500,
    });
  }

  if ('message' in agencies) {
    return new Response(JSON.stringify({ error: agencies.message }), {
      status: 500,
    });
  }

  if (!agencies || !agencies.body) {
    return new Response(JSON.stringify({ error: "No statuses found." }), {
      status: 404,
    });
  }

  return new Response(JSON.stringify(agencies), {
    status: 200,
  });
}


async function getAgencies() {
  const session = await getCurrentSessionFromServer()
  const bearer = session.tokens?.idToken?.toString()
  if (!bearer) {
    return new Error(INTERNAL__NotAuthorizedError)
  }

  const res = await validatedFetch({
    url: env.BACKEND_OTP_FEED_STATUS_URL,
    schema: otpFeedStatusesResponseSchema,
    init: {
      method: "GET",
      headers: {
        "Authorization": `Bearer ${bearer}`,
      }
    }
  });

  return res
}

